// ============================================================================
// ============================================================================
// ============================================================================
// ==                                                                        ==
// == Name    : TheEmuLib.Emu_Ripple_RA.fsh                                  ==
// == Type    : Fragment shader                                              ==
// == Version : 1.0.15 (2017/01/18)                                          ==
// == Creator : TheEmu © TheEmu 2017, Some Rights Reserved                   ==
// == Licence : Creative Commons Attribution-ShareAlike 4.0                  ==
// ==           http://creativecommons.org/licences/by-sa/4.0                ==
// ==                                                                        ==
// == Purpose: To apply a "ripple" to the polar coordinates of the points in ==
// == an image with the ripple being a function of their (r,a) coordinates.  ==
// ==                                                                        ==
// == Description:  A  source image is modified by applying a time dependant ==
// == wobbling motion. The ripple at any point comprises a sum of up to four ==
// == terms added to its normalised coordinates, each term being of the form ==
// ==                                                                        ==
// ==        K0 * Ar * cos ( T*Fr - B*r - C*a + Dx ) + K1                    ==
// ==        K0 * Aa * cos ( T*Fa - B*a - C*r + Dy ) + K1                    ==
// ==                                                                        ==
// == where T is time, (r,a) the normalised coordinates and A, B, C, D, F, K ==
// == are ripple parameters supplied as shader inputs.  The negation of some ==
// == terms involving B, C, D and K is done so that positive values of these ==
// == parameters correspond to movement in a positive direction.             ==
// ==                                                                        ==
// == Static distortion of the source image is obtained using a frequency of ==
// == of 0.0 for all components of the ripple.                               ==
// ==                                                                        ==
// == This file is a member of The Emu's shader library.                     ==
// ==                                                                        ==
// == ====================================================================== ==
// ==                                                                        ==
// == Update history:                                                        ==
// ==                                                                        ==
// ==   2016/12/17 - v1.0.0 - Initial version 0 - from Emu_Wobble_RA.fsh     ==
// ==   2016/12/17 - v1.0.1 - Borders replaced by edge coordinates.          ==
// ==   2016/12/18 - v1.0.2 - Added edge_drops to drop the unrippled region. ==
// ==   2016/12/20 - v1.0.3 - Added overall ripple scale and offset.         ==
// ==   2016/12/21 - v1.0.4 - Changed signs of B and C terms.                ==
// ==   2017/01/05 - v1.0.5 - Renamed all Wobble shaders to be Ripple.       ==
// ==   2017/01/05 - v1.0.7 - Bug fix for edge_hold logic.                   ==
// ==   2017/01/05 - v1.0.7 - Increased commonality with Ripple_XY.          ==
// ==   2017/01/06 - v1.0.8 - mix(vec4,vec4,bvec4) not allowed by NVIDIA.    ==
// ==   2017/01/06 - v1.0.9 - Interchanged edge_modes 1 and 2.               ==
// ==   2017/01/06 - v1.0.10 - Fixed bug in edge smearing logic.             ==
// ==   2017/01/15 - v1.0.11 - Fixed bug in edge drop logic.                 ==
// ==   2017/01/15 - v1.0.12 - Added support for scale and hotspot.          ==
// ==   2017/01/15 - v1.2.13 - Added an edge spill option.                   ==
// ==   2017/01/16 - v1.2.14 - Corrected overall sign of the ripple.         ==
// ==   2017/01/18 - v1.2.15 - Corrected error in edge logic                 ==
// ==                                                                        ==
// ============================================================================
// ============================================================================
// ============================================================================

// ============================================================================
// == Standard shader inputs ==================================================
// ============================================================================

uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

// The image that is to be manipulated.

uniform sampler2D iChannel0;

// ============================================================================
// == Imports from TheEmuLib ==================================================
// ============================================================================
//
// The GLSL shader language currently provides no mechanism for importing  any
// elements that are defined in other modules, not even C's crude source level
// #include mechanism. In the absence of anything better TheEmuLib handles any
// imports by manualy copying relevent utility code snippets from the  sources
// in the Shader Lib.Inc directory. This is very crude but I have attempted to
// be systematic in the way in which this is presented in the library sources.
//
// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

// Macros from TheEmuLib.Emu_Basic_Constants.lib.src

#define PI  3.1415926535897932384626433832795
#define TAU 6.2831853071795864769252867665590

// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

// Macros from TheEmuLib.Emu_Common_Utilities.lib.src

#define EMU_DEFAULT(type,x,default_value) ( (x==type(0.0)) ? (default_value) : (x) )

// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

// Functions from TheEmuLib.Emu_Coordinate_Normalisation.lib.src

#define EMU_NORMALISE_TO_WINDOW_1(xy,wsize) (     (xy)/(wsize) )
#define EMU_NORMALISE_TO_WINDOW_2(xy,wsize) ( 2.0*(xy)/(wsize) - 1.0 )

#define EMU_NORMALISE_TO_WINDOW_3(xy,c1,c2) ( EMU_NORMALISE_TO_WINDOW_1(xy-c1,c2-c1) )
#define EMU_NORMALISE_TO_WINDOW_4(xy,c1,c2) ( EMU_NORMALISE_TO_WINDOW_2(xy-c1,c2-c1) )

#define EMU_DENORMALISE_TO_WINDOW_1(uv,size)  (      (uv)        * (size) )
#define EMU_DENORMALISE_TO_WINDOW_2(uv,size)  ( ( (uv)*0.5+0.5 ) * (size) )

#define EMU_DENORMALISE_TO_WINDOW_3(uv,c1,c2) ( EMU_DENORMALISE_TO_WINDOW_1(uv,c2-c1)+c1 )
#define EMU_DENORMALISE_TO_WINDOW_4(uv,c1,c2) ( EMU_DENORMALISE_TO_WINDOW_2(uv,c2-c1)+c1 )

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

vec2 Emu_Normalise_to_Window ( vec2 xy )
 { return EMU_NORMALISE_TO_WINDOW_1 ( xy, u_WindowSize.xy );
 }

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

vec2 Emu_Normalise_to_Window ( vec2 xy, vec2 xy0, vec2 xy1 )
 { return EMU_NORMALISE_TO_WINDOW_3 ( xy, xy0, xy1 );
 }

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

vec2 Emu_Denormalise_to_Window ( vec2 uv, vec2 xy0, vec2 xy1 )
 { return EMU_DENORMALISE_TO_WINDOW_3 ( uv, xy0, xy1 );
 }

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

vec2 Emu_Denormalise_to_Window_Centered ( vec2 uv, vec2 window )
 { return EMU_DENORMALISE_TO_WINDOW_2 ( uv, window );
 }

// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

// Functions from TheEmuLib.Emu_Polar_Coordinates.lib.src

vec2 Emu_XY_To_Polar_Coordinates ( vec2 xy )
 { return vec2 ( length(xy), atan(xy.y,xy.x) );
 }

vec2 Emu_Polar_Coordinates_to_XY ( vec2 ra )
 { return vec2 ( cos(ra[1]), sin(ra[1]) ) * ra[0];
 }

// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

// Macros from TheEmuLib.Emu_Boolean_Vector_Ops.lib.src

#define Emu_bvec4_or(A,B)  bvec4 ( A[0]||B[0], A[1]||B[1], A[2]||B[2], A[3]||B[3] )
#define Emu_bvec4_and(A,B) bvec4 ( A[0]&&B[0], A[1]&&B[1], A[2]&&B[2], A[3]&&B[3] )
#define Emu_bvec4_xor(A,B) bvec4 ( A[0]^^B[0], A[1]^^B[1], A[2]^^B[2], A[3]^^B[3] )

#define Emu_bvec2_or(A,B)  bvec2 ( A[0]||B[0], A[1]||B[1] )
#define Emu_bvec2_and(A,B) bvec2 ( A[0]&&B[0], A[1]&&B[1] )
#define Emu_bvec2_xor(A,B) bvec2 ( A[0]^^B[0], A[1]^^B[1] )

// ============================================================================
// == Shader specific inputs ==================================================
// ============================================================================

uniform vec2 Emu_Ripple_RA_scale;
uniform vec2 Emu_Ripple_RA_hotspot;

vec2 scale   = EMU_DEFAULT ( vec2, Emu_Ripple_RA_scale,   vec2(1.0) );
vec2 hotspot = EMU_DEFAULT ( vec2, Emu_Ripple_RA_hotspot, vec2(0.0) );

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// By default the whole area of the source image is rippled, but this can be
// overridden by using the following parameter to limit the to wobbling to a
// particular region.  The  edges parameter defines the region that is to be
// rippled by specifying the low and high limits of normalised coordinates.

uniform vec4 Emu_Ripple_RA_edges;

vec4 edges =
  EMU_DEFAULT ( vec4, Emu_Ripple_RA_edges,vec4(0.0,1.0,0.0,1.0) );

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// The effect of the wobbling at the edges of the rippled area can result in
// discontinuities in the generated image.  By default any tears between the
// rippled area and its suroundings represented by fully transparent  black,
// i.e. they become holes through which any background image can be seen.
//
// The default behaviour of producing tears at the edges can be overidden by
// using the edge_mode parameter, which may have the values:-
//
//   0 - edge_tears - Default, produces "tears" at the edge
//   1 - edge_hold1 - Ripples at edge and orthogonal to it held to 0.0
//   2 - edge_hold2 - Ripples at edge and parallel to it held to 0.0
//   3 - edge_hold3 - Combination of edge_hold1 and edge_hold2
//   4 - edge_wraps - Ripple wraps round to the opposite side
//   5 - edge_smear - Any tear is filled with the edge's colour
//   6 - edge_selfs - The source image itself fills edge tears
//
// In mode 0 any background image is seen through a tear.
// In mode 1 ripples parallel to the edge fade out near the edge
// In mode 2 ripples orthogonal to the edge fade out near the edge
// In mode 3 ripples of either type fade out near the edge
// In mode 4 ripples from opposing edges wrap round, eliminating tears
// In mode 5 the edge, in effect, is smeared though a tear.
// In mode 6 the source image acts as a background in a tear
//
// If 10 is added to the edge mode then anything outside the rippled area is
// rendered transparent rather than retaining its original colour.
//
// If 20 is added to the edge mode then ripples may "spill" over the rippled
// area's boundary into the unrippled area beyond it.
//
// The edge mode is specified separately for all four edges using  the  vec4
// input parameter Emu_Ripple_RA_edge_mode where the elements give the modes
// for the edges at X=0, X=1, Y=0 and Y=1 in that order.  Unfortunately  the
// iStripper program does not support passing an ivec4 from the  scene  file
// so we have to use a vec4 and ignore any fractional part.

uniform vec4 Emu_Ripple_RA_edge_mode;

ivec4 edge_mode1 = ivec4 ( mod(Emu_Ripple_RA_edge_mode/10.0,10.0) );
ivec4 edge_mode2 = ivec4 ( mod(Emu_Ripple_RA_edge_mode,10.0) );

bvec4 edge_spill = equal ( edge_mode1 & 2, ivec4(2) );
bvec4 edge_drops = equal ( edge_mode1 & 1, ivec4(1) );

bvec4 edge_undef = greaterThan ( edge_mode2, ivec4(6) );

bvec4 edge_selfs = equal ( edge_mode2, ivec4(6) );
bvec4 edge_smear = equal ( edge_mode2, ivec4(5) );
bvec4 edge_wraps = equal ( edge_mode2, ivec4(4) );
bvec4 edge_hold3 = equal ( edge_mode2, ivec4(3) );

bvec4 edge_hold2 = Emu_bvec4_or ( equal(edge_mode2,ivec4(2)), edge_hold3 );
bvec4 edge_hold1 = Emu_bvec4_or ( equal(edge_mode2,ivec4(1)), edge_hold3 );
bvec4 edge_tears = Emu_bvec4_or ( equal(edge_mode2,ivec4(0)), edge_undef );

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// Overall scale and offset for the ripple, the K0 and K1 parameters refered
// to in the initial description of the shader. K0 is not strictly necessary
// as the same effect could be obtained in the scene file by scaling each of
// the A terms, but it is convenient for the shader user to be able to tweak
// the overall scale rather than changing all of the A parameters.  K1  just
// adds a constant offset to the ripple,  the  same effect could be obtained
// in the scene file by altering the position of the object being rippled.

uniform vec2 Emu_Ripple_RA_KK;  // Ripple overall scale and offset

vec2 KK = EMU_DEFAULT ( vec2, Emu_Ripple_RA_KK, vec2(1.0,0.0) );

// The ripple parameters for each component of the ripple. Any parameter not
// specified defaults to zero. The spatial frequency terms are the number of
// ripples per unit distance.

uniform vec4 Emu_Ripple_RA_Ar;  // Amplitudes
uniform vec4 Emu_Ripple_RA_Aa;

uniform vec4 Emu_Ripple_RA_Br;  // Spatial frequencies (X component terms)
uniform vec4 Emu_Ripple_RA_Ba;

uniform vec4 Emu_Ripple_RA_Cr;  // Spatial frequencies (Y component terms)
uniform vec4 Emu_Ripple_RA_Ca;

uniform vec4 Emu_Ripple_RA_Dr;  // Phases in fractions of a cycle
uniform vec4 Emu_Ripple_RA_Da;

uniform vec4 Emu_Ripple_RA_Fr;  // Frequencies in Hz
uniform vec4 Emu_Ripple_RA_Fa;

// Ripple parameters as used in the body of the shader.

#define Ar ( Emu_Ripple_RA_Ar )
#define Aa ( Emu_Ripple_RA_Aa )

#define Br ( Emu_Ripple_RA_Br * TAU )
#define Ba ( Emu_Ripple_RA_Ba * TAU )

#define Cr ( Emu_Ripple_RA_Cr * TAU )
#define Ca ( Emu_Ripple_RA_Ca * TAU )

#define Dr ( Emu_Ripple_RA_Dr * TAU )
#define Da ( Emu_Ripple_RA_Da * TAU )

#define Fr ( Emu_Ripple_RA_Fr * TAU )
#define Fa ( Emu_Ripple_RA_Fa * TAU )

// ============================================================================
// == The shader's major function =============================================
// ============================================================================

vec4 Emu_Ripple_RA ( vec2 uv0 )
 {
   // Normalise to the region that is to be rippled.

   vec2 ra0 = Emu_XY_To_Polar_Coordinates ( 2.0*uv0 - 1.0 );
   ra0[1] = ra0[1]/TAU + 0.5;

   vec2 pq = Emu_Normalise_to_Window ( ra0, edges.xz, edges.yw );

   // Determine which edges affect the ripple at this point.

   bvec4 edge = bvec4 ( pq.x < 0.5, pq.x > 0.5,
                        pq.y < 0.5, pq.y > 0.5
                      );

   // Update selected edge flags to reflect the nearest edges.

   edge_drops = Emu_bvec4_and ( edge, edge_drops );
   edge_wraps = Emu_bvec4_and ( edge, edge_wraps );
   edge_spill = Emu_bvec4_and ( edge, edge_spill );

   bool edge_out_0 = pq != fract(pq);

   // If outside the rippled area then no ripple is applied and we just have
   // to return the colour corresponding to the unrippled coordinates unless
   // the outer region is to be dropped when we return transparent.  If  the
   // edge spills mode is active then this decision has to be delayed  until
   // we know whether the current pixel is in a spill area or not.

   if ( edge_out_0 && ! any(edge_spill) )
    { bool drop = any(edge_drops);
      return mix ( texture2D(iChannel0,uv0), vec4(0.0), vec4(drop) );
    }

   // Update the remaining edge flags to reflect the nearest edges.

   edge_tears = Emu_bvec4_and ( edge, edge_tears );
   edge_hold1 = Emu_bvec4_and ( edge, edge_hold1 );
   edge_hold2 = Emu_bvec4_and ( edge, edge_hold2 );
   edge_smear = Emu_bvec4_and ( edge, edge_smear );
   edge_selfs = Emu_bvec4_and ( edge, edge_selfs );

   // Calculate the four components for both the x and y ripples.
   // See comments at top of shader for why B and C are negated.

   vec4 w1 = Ar * cos ( u_Elapsed*Fr - Br*pq.x - Cr*pq.y + Dr );
   vec4 w2 = Aa * cos ( u_Elapsed*Fa - Ba*pq.x - Ca*pq.y + Da );

   // Apply overall ripple scale and offset parameters.

   w1 = KK[0]*w1 + KK[1];
   w2 = KK[0]*w2 + KK[1];

   // Sum the four components of the wobbling for each axis.

   vec2 ww = vec2 ( dot ( vec4(1.0), w1 ),
                    dot ( vec4(1.0), w2 )
                  );

   // Optionaly force the ripples to to be zero at the edges. This is done
   // by multiplying them by a function, sin, that is zero at the edges.
   //
   // Ripples orthogonal to the edges are controlled by edge_hold1.
   // Ripples parallel to the edges are controlled by edge_hold2.

   vec2 ff = sin(pq*PI);

   if ( any(edge_hold1.xy) ) ww.x *= ff.x;
   if ( any(edge_hold1.zw) ) ww.y *= ff.y;

   if ( any(edge_hold2.xy) ) ww.y *= ff.x;
   if ( any(edge_hold2.zw) ) ww.x *= ff.y;

   // Apply the ripple to the normalised coordinates.

   pq -= ww;

   // Determine if there is a tear or a spill and, when there is a tear,
   // whether the source image is to be visible through it or not.

   bvec2 tear_or_spill = notEqual ( pq, fract(pq) );

   bvec4 edge_out_1 = bvec4 ( pq.x < 0.0, pq.x > 1.0,
                              pq.y < 0.0, pq.y > 1.0 );

   bvec4 is_spilled = bvec4 ( ww.x < 0.0, ww.x > 0.0,
                              ww.y < 0.0, ww.y > 0.0 );

   is_spilled = Emu_bvec4_and ( is_spilled, edge_out_1 );

   edge_wraps = Emu_bvec4_and ( edge_wraps, bvec4(!edge_out_0) );

   edge_tears = Emu_bvec4_and ( edge_tears, tear_or_spill.xxyy );
   edge_selfs = Emu_bvec4_and ( edge_selfs, tear_or_spill.xxyy );

   edge_spill = Emu_bvec4_and ( edge_spill, is_spilled      );
   edge_tears = Emu_bvec4_and ( edge_tears, not(is_spilled) );
   edge_selfs = Emu_bvec4_and ( edge_selfs, not(is_spilled) );

   // Perform any edge wrap or smear where it is enabled.

   if ( any(edge_wraps.xy) ) pq.x = fract(pq.x);
   if ( any(edge_wraps.zw) ) pq.y = fract(pq.y);

   if ( any(edge_smear.xy) ) pq.x = clamp ( pq.x, 0.0, 1.0 );
   if ( any(edge_smear.zw) ) pq.y = clamp ( pq.y, 0.0, 1.0 );

   // Renormalise to the dimensions of the whole image.

   vec2 st = Emu_Denormalise_to_Window ( pq, edges.xz, edges.yw );

   st = Emu_Polar_Coordinates_to_XY ( vec2(st.s,st.t*TAU-PI) );
   st = Emu_Denormalise_to_Window_Centered ( st, vec2(1.0) );

   // Get the colour for the pixel.

   vec4  colour;
   float opacity = 1.0;

   if ( any(edge_out_1) )
    {
      if ( any(Emu_bvec4_and(edge_out_1,edge_selfs)) ) st = uv0;

      if ( any(Emu_bvec4_and(edge_out_1,edge_drops)) ) opacity = 0.0;

      if ( ! edge_out_0 )
       { if ( any(Emu_bvec4_and(edge_out_1,edge_tears)) ) opacity = 0.0;
         if ( any(Emu_bvec4_and(edge_out_1,edge_smear))
           || any(Emu_bvec4_and(edge_out_1,edge_selfs)) ) opacity = 1.0;
       }

      if ( any(Emu_bvec4_and(edge_out_1,edge_wraps)) ) opacity = 1.0;

    }

   colour = texture2D ( iChannel0, st ) * opacity;

   // Debug code. Uncomment individual lines to see what parts of  the
   // image are controlled by the corresponding flag. If checking more
   // than one flag at a time then use different colours for each.  If
   // details are obscured by the normal image then forcing the colour
   // first will remove it and just show the debugging data.

   // colour = vec4(0.1,0.1,0.1,1.0);        // Optionaly force grey.
   //                                        //
   // if (     edge_out_0  ) colour.r = 1.0; // Colour selected areas
   // if ( any(edge_out_1) ) colour.g = 1.0; // depending on which of
   // if ( any(edge_drops) ) colour.b = 1.0; // the control flags are
   // if ( any(edge_tears) ) colour.b = 1.0; // set.  It is best when
   // if ( any(edge_wraps) ) colour.b = 1.0; // different colours are
   // if ( any(edge_spill) ) colour.b = 1.0; // used for each area.
   // if ( any(edge_smear) ) colour.b = 1.0; //
   // if ( any(edge_selfs) ) colour.b = 1.0; //

   return colour;

 }

// ============================================================================
// == The shader's main routine ===============================================
// ============================================================================

void main ( void )
 {
   // Get the normalised coordinates of the current point.

   vec2 uv = Emu_Normalise_to_Window ( gl_FragCoord.xy );

   uv = ( uv - hotspot ) / scale + hotspot;

   // Apply the ripple and update the shader's outputs.

   gl_FragColor = Emu_Ripple_RA(uv) * gl_Color;

 }

// ============================================================================
